<?php
$logFile = 'access.log';

if (isset($_POST['clear_log'])) {
    if (file_exists($logFile)) {
        file_put_contents($logFile, ""); 
        echo json_encode(['status' => 'success']);
    }
    exit;
}

if (isset($_GET['ajax'])) {
    renderContent($logFile);
    exit;
}

function renderContent($logFile) {
    if (!file_exists($logFile)) {
        echo json_encode(['total' => 0, 'bot' => 0, 'human' => 0, 'rows' => ""]);
        return;
    };

    $logs = file($logFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    $logs = array_reverse($logs);

    $totalVisitor = count($logs);
    $totalHuman = 0;
    $totalBot = 0;
    $tableRows = "";

    foreach ($logs as $line) {
        $parts = explode(' | ', $line);
        if (count($parts) >= 5) {
            $date    = trim($parts[0]);
            $info    = trim($parts[1]);
            $ip      = trim($parts[2]);
            $country = trim($parts[3]);
            $ua      = trim($parts[4]);

            $isHuman = (strtolower($info) == 'human');
            if ($isHuman) {
                $totalHuman++;
                $class = 'status-human';
            } else {
                $totalBot++;
                $class = 'status-bot';
            }

            $tableRows .= "<tr>
                <td>$date</td>
                <td><span class='$class'>$info</span></td>
                <td>$ip</td>
                <td><span class='country-code'>$country</span></td>
                <td class='ua-dim'>$ua</td>
            </tr>";
        }
    }

    echo json_encode([
        'total' => $totalVisitor,
        'bot' => $totalBot,
        'human' => $totalHuman,
        'rows' => $tableRows
    ]);
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <title>Real-time Security Dashboard</title>
    <style>
        body { background-color: #0c0c0c; color: #e0e0e0; font-family: 'Consolas', monospace; padding: 20px; }
        .header-flex { display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px; }
        .stats-container { display: flex; gap: 15px; margin-bottom: 25px; }
        .card { flex: 1; border: 1px solid #333; border-left: 4px solid #ff0000; background: #161616; padding: 15px; border-radius: 4px; }
        .card.human { border-left-color: #00ff00; }
        .card-label { font-size: 11px; color: #888; text-transform: uppercase; }
        .card-value { font-size: 28px; font-weight: bold; text-align: right; margin-top: 5px; }
        
        .btn-delete { 
            background: #441111; color: #ff8888; border: 1px solid #662222; 
            padding: 8px 15px; cursor: pointer; font-family: 'Consolas', monospace;
            border-radius: 3px; transition: 0.3s;
        }
        .btn-delete:hover { background: #aa0000; color: white; }

        table { width: 100%; border-collapse: collapse; }
        th { text-align: left; padding: 12px; color: #888; font-size: 12px; border-bottom: 2px solid #222; }
        td { padding: 10px; border-bottom: 1px solid #1a1a1a; font-size: 12px; }
        .status-human { color: #00ff00; font-weight: bold; }
        .status-bot { color: #ff3e3e; font-weight: bold; }
        .country-code { background: #333; padding: 2px 6px; border-radius: 3px; font-size: 10px; }
        .ua-dim { color: #aaa; font-size: 11px; max-width: 300px; overflow: hidden; text-overflow: ellipsis; white-space: nowrap; }
    </style>
</head>
<body>

    <div class="header-flex">
        <h2 style="margin:0; color: #fff;">SECURITY DASHBOARD</h2>
        <button class="btn-delete" onclick="clearLogs()">EMPTY LOG FILE</button>
    </div>

    <div class="stats-container">
        <div class="card"><div class="card-label">Total Visits</div><div class="card-value" id="count-total">0</div></div>
        <div class="card"><div class="card-label">Bot / Blocked</div><div class="card-value" id="count-bot" style="color: #ff3e3e;">0</div></div>
        <div class="card human"><div class="card-label">Human</div><div class="card-value" id="count-human" style="color: #00ff00;">0</div></div>
    </div>

    <h3>⇅ VISITOR LOG <small id="status-update" style="font-size: 10px; color: #444; margin-left: 10px;">Updating...</small></h3>

    <table>
        <thead>
            <tr>
                <th>DATE</th><th>INFO</th><th>IP ADDRESS</th><th>CO</th><th>USER AGENT</th>
            </tr>
        </thead>
        <tbody id="log-table-body"></tbody>
    </table>

    <script>
        function updateDashboard() {
            fetch('?ajax=1')
                .then(response => response.json())
                .then(data => {
                    document.getElementById('count-total').innerText = data.total;
                    document.getElementById('count-bot').innerText = data.bot;
                    document.getElementById('count-human').innerText = data.human;
                    document.getElementById('log-table-body').innerHTML = data.rows;
                    document.getElementById('status-update').innerText = 'Last updated: ' + new Date().toLocaleTimeString();
                })
                .catch(error => console.error('Error:', error));
        }

        function clearLogs() {
            if (confirm('Apakah Anda yakin ingin menghapus semua log? Tindakan ini tidak bisa dibatalkan.')) {
                const formData = new FormData();
                formData.append('clear_log', '1');

                fetch('', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(result => {
                    if (result.status === 'success') {
                        updateDashboard(); 
                    }
                })
                .catch(error => console.error('Error:', error));
            }
        }

        updateDashboard();
        setInterval(updateDashboard, 3000);
    </script>
</body>
</html>